/*
* Creation date : 
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains implementation of 
*  different functions. 
*
* \version CE2_Utils.c#1:csrc:1
* \author Yermalayeu Ihar
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Include Files ***********************/

#include "CE2_Utils.h"
#include "LLF_Utils.h"
#include "LLF_RSA_SCHEMES.h"

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Global Data *************************/
/************************ Private function prototype **********/
/************************ Private Functions *******************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
* CE2_MulNum
*
*  @param Buf1_ptr [in] - A pointer to buffer with first big-endian number;
*  @param Buf1SizeInBits [in] - The size of the first number in bits;
*  @param Buf2_ptr [in] - A pointer to buffer with second big-endian number; 
*  @param Buf2SizeInBits [in] - The size of the second number in bits;
*  @param Res_ptr [out] - A pointer to the buffer with output big-endian number; 
*  @param ResSizeInBytes_ptr [in/out] - The size of the output number in bytes.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_UTILS_INVALID_POINTER_ERROR
*
* \brief \b 
* Description:
*  This function multiplicate two big number and store result in output buffer;
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_MulNum 
*     for big number multiplication with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_MulNum( 
                                DxUint32_t *Buf1_ptr, 
                                DxUint32_t Buf1SizeInBits,
                                DxUint32_t *Buf2_ptr, 
                                DxUint32_t Buf2SizeInBits,
                                DxUint32_t *Res_ptr, 
                                DxUint32_t *ResSizeInBytes_ptr)
{
  if (Buf1_ptr == DX_NULL)
    return CE2_UTILS_INVALID_POINTER_ERROR;

  if (Buf2_ptr == DX_NULL)
    return CE2_UTILS_INVALID_POINTER_ERROR;

  if (Res_ptr == DX_NULL)
    return CE2_UTILS_INVALID_POINTER_ERROR;

  if (ResSizeInBytes_ptr == DX_NULL)
    return CE2_UTILS_INVALID_POINTER_ERROR;

  return LLF_MulNum(Buf1_ptr, Buf1SizeInBits,Buf2_ptr, 
    Buf2SizeInBits, Res_ptr, ResSizeInBytes_ptr);
} /* End of CE2_MulNum */

/**
****************************************************************
* Function Name: 
* CE2_GetBufferEffectiveSizeInBits
*
*  @param Buffer_ptr [in] - A pointer to buffer with big-endian number;
*  @param BufferSizeInBytes [in] - The size of the buffer;
*  @param EffectiveSizeInBits_ptr [out] - A output pointer with effective 
*                                         size in bits.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_UTILS_INVALID_POINTER_ERROR
*
* \brief \b 
* Description:
*  Get effective size in bits for given buffer with big number in 
*  big-endian format.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_GetBufferEffectiveSizeInBits 
*     for getting effective buffer size with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_GetBufferEffectiveSizeInBits(
  DxUint8_t  *Buffer_ptr,
  DxUint16_t BufferSizeInBytes,
  DxUint32_t *EffectiveSizeInBits_ptr)
{
  if (Buffer_ptr == DX_NULL)
    return CE2_UTILS_INVALID_POINTER_ERROR;

  if (EffectiveSizeInBits_ptr == DX_NULL)
    return CE2_UTILS_INVALID_POINTER_ERROR;

  return LLF_GetBufferEffectiveSizeInBits(Buffer_ptr, BufferSizeInBytes,
    EffectiveSizeInBits_ptr);
} /* End of CE2_GetBufferEffectiveSizeInBits */

/**
****************************************************************
* Function Name: 
* CE2_GetDEREncodingLength
*
*  @param hashFunc [in] - RSA hash function
*  @param pTLength [out] - Returns the DER encoding length (tLen), 
*                          as explained in the EMSA-PKCS1-v1_5
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*   - CE2_UTILS_INVALID_POINTER_ERROR
*
* \brief \b 
* Description:
*  Returns the DER encoding length (tLen), as explained in the EMSA-PKCS1-v1_5
*
*  \b 
* Algorithm:
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_GetDEREncodingLength(CE2_RSA_HASH_OpMode_t hashFunc,
                                              DxUint32_t			      *pTLength)
{
  if (pTLength == DX_NULL)
    return CE2_UTILS_INVALID_POINTER_ERROR;

  return LLF_DX_RSA_GetDEREncodingLength(hashFunc, pTLength);
} /* End of CE2_GetDEREncodingLength */
